/*
 * Decompiled with CFR 0.152.
 */
package org.bff.javampd.server;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.Socket;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import org.bff.javampd.MPDException;
import org.bff.javampd.command.MPDCommand;
import org.bff.javampd.server.MPDConnectionException;
import org.bff.javampd.server.MPDSecurityException;
import org.bff.javampd.server.ResponseProperties;
import org.bff.javampd.server.ServerProperties;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MPDSocket {
    private static final Logger LOGGER = LoggerFactory.getLogger(MPDSocket.class);
    private Socket socket;
    private BufferedReader reader;
    private final ResponseProperties responseProperties;
    private final ServerProperties serverProperties;
    private final String encoding;
    private String lastError;
    private String version;
    private final String server;
    private final int port;
    private boolean closed;
    private static final int TRIES = 3;

    public MPDSocket(InetAddress server, int port, int timeout) {
        this.server = server.getHostAddress();
        this.port = port;
        this.responseProperties = new ResponseProperties();
        this.serverProperties = new ServerProperties();
        this.encoding = this.serverProperties.getEncoding();
        this.connect(timeout);
    }

    private synchronized void connect(int timeout) {
        this.connectSocket(timeout);
    }

    private void readVersion() {
        String line;
        try {
            line = this.reader.readLine();
        }
        catch (IOException e) {
            throw new MPDConnectionException(e);
        }
        if (line == null || !this.isResponseOK(line)) {
            throw new MPDConnectionException("Command from server: " + (line == null ? "null" : MPDSocket.stripResponse(this.responseProperties.getError(), line)));
        }
        this.version = MPDSocket.stripResponse(this.responseProperties.getOk(), line).trim();
    }

    private void connectSocket(int timeout) {
        LOGGER.debug("attempting to connect socket to {} with timeout of {}", (Object)this.server, (Object)timeout);
        this.socket = this.createSocket();
        InetSocketAddress socketAddress = new InetSocketAddress(this.server, this.port);
        try {
            this.socket.connect(socketAddress, timeout);
            this.setReader(new BufferedReader(new InputStreamReader(this.socket.getInputStream(), this.encoding)));
            this.readVersion();
        }
        catch (Exception ioe) {
            LOGGER.error("failed to connect socket to {}", (Object)this.server);
            throw new MPDConnectionException(ioe);
        }
    }

    protected void setReader(BufferedReader reader) {
        this.reader = reader;
    }

    protected Socket createSocket() {
        return new Socket();
    }

    public synchronized Collection<String> sendCommand(MPDCommand command) {
        this.checkConnection();
        int count = 0;
        while (count < 3) {
            try {
                return this.sendBytes(MPDSocket.convertCommand(command.getCommand(), command.getParams()));
            }
            catch (MPDException mpdException) {
                MPDSocket.logCommandError(command, mpdException);
                throw mpdException;
            }
            catch (Exception ex) {
                MPDSocket.logCommandError(command, ex);
                try {
                    this.connect();
                }
                catch (Exception exc) {
                    LOGGER.error("Unable to connect to {} on port {}", this.server, this.port, exc);
                }
                LOGGER.warn("Retrying command {} for the {} time", (Object)command.getCommand(), (Object)(++count));
            }
        }
        LOGGER.error("Unable to send command {} after {} tries", (Object)command, (Object)3);
        throw new MPDConnectionException("Unable to send command " + command);
    }

    private static void logCommandError(MPDCommand command, Exception se) {
        LOGGER.error("Error from: {}", (Object)command.getCommand(), (Object)se);
        for (String str : command.getParams()) {
            LOGGER.error("\tparam: {}", (Object)str);
        }
    }

    private synchronized void connect() {
        this.connect(0);
    }

    private boolean isResponseOK(String line) {
        try {
            if (line.startsWith(this.responseProperties.getOk()) || line.startsWith(this.responseProperties.getListOk())) {
                return true;
            }
        }
        catch (Exception e) {
            LOGGER.error("Could not determine if response is ok", e);
        }
        return false;
    }

    private boolean isResponseError(String line) {
        if (line.startsWith(this.responseProperties.getError())) {
            this.lastError = line.substring(this.responseProperties.getError().length()).trim();
            return true;
        }
        return false;
    }

    private static String stripResponse(String response, String line) {
        return line.substring(response.length());
    }

    private static String convertCommand(String command) {
        return MPDSocket.convertCommand(command, new ArrayList<String>());
    }

    private static String convertCommand(String command, List<String> params) {
        StringBuilder sb = new StringBuilder(command);
        for (String param : params) {
            param = param.replaceAll("\"", "\\\\\"");
            sb.append(" \"").append(param).append("\"");
        }
        return sb.append("\n").toString();
    }

    public synchronized void sendCommands(List<MPDCommand> commandList) {
        StringBuilder sb = new StringBuilder(MPDSocket.convertCommand(this.serverProperties.getStartBulk()));
        for (MPDCommand command : commandList) {
            sb.append(MPDSocket.convertCommand(command.getCommand(), command.getParams()));
        }
        sb.append(MPDSocket.convertCommand(this.serverProperties.getEndBulk()));
        this.checkConnection();
        try {
            this.sendBytes(sb.toString());
            String line = this.reader.readLine();
            while (line != null) {
                if (!this.isResponseOK(line)) {
                    LOGGER.warn("some command from a command list failed: {}", (Object)line);
                }
                if (this.reader.ready()) {
                    line = this.reader.readLine();
                    LOGGER.warn("unexpected response line {}", (Object)line);
                    continue;
                }
                line = null;
            }
        }
        catch (MPDSecurityException se) {
            LOGGER.error("Response Error from command list", se);
            throw se;
        }
        catch (Exception e) {
            LOGGER.error("Response Error from command list", e);
            commandList.forEach(s2 -> LOGGER.error(s2.getCommand()));
            throw new MPDConnectionException(e.getMessage(), e);
        }
    }

    private List<String> sendBytes(String command) throws IOException {
        LOGGER.debug("start command: {}", (Object)command);
        ArrayList<String> response = new ArrayList<String>();
        this.writeToStream(command);
        String inLine = this.reader.readLine();
        LOGGER.debug("first response line is: {}", (Object)inLine);
        while (inLine != null) {
            if (this.isResponseOK(inLine)) {
                LOGGER.debug("the response was ok");
                break;
            }
            if (this.isResponseError(inLine)) {
                if (this.lastError.contains("you don't have permission")) {
                    throw new MPDSecurityException(this.lastError, command);
                }
                LOGGER.error("Got error from command {}", (Object)command);
                throw new MPDConnectionException(this.lastError);
            }
            response.add(inLine);
            inLine = this.reader.readLine();
        }
        response.forEach(LOGGER::debug);
        return response;
    }

    private void checkConnection() {
        boolean connected;
        if (this.closed) {
            throw new MPDConnectionException("Close has been called on MPD.  Create a new MPD.");
        }
        if (!this.socket.isConnected()) {
            LOGGER.warn("socket hasn't been connected yet");
            connected = false;
        } else if (!this.socket.isClosed()) {
            connected = this.checkPing();
        } else {
            LOGGER.warn("socket is closed");
            connected = false;
        }
        if (!connected) {
            LOGGER.warn("we've lost connectivity, attempting to connect");
            try {
                this.connect();
            }
            catch (Exception e) {
                throw new MPDConnectionException("Connection to server lost: " + e.getMessage(), e);
            }
        }
    }

    public void close() {
        this.closed = true;
        if (!this.socket.isClosed()) {
            try {
                this.reader.close();
            }
            catch (IOException e) {
                throw new MPDConnectionException("Unable to close socket", e);
            }
            try {
                this.socket.close();
            }
            catch (IOException e) {
                throw new MPDConnectionException("Unable to close socket", e);
            }
        }
    }

    private void writeToStream(String command) throws IOException {
        this.socket.getOutputStream().write(command.getBytes(this.serverProperties.getEncoding()));
    }

    public String getVersion() {
        return this.version;
    }

    private boolean checkPing() {
        boolean connected = true;
        try {
            this.writeToStream(MPDSocket.convertCommand(this.serverProperties.getPing()));
            String inLine = this.reader.readLine();
            if (!this.isResponseOK(inLine)) {
                connected = false;
            }
        }
        catch (Exception e) {
            connected = false;
            LOGGER.error("lost socket connection", e);
        }
        return connected;
    }
}

